/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include <udm_config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <errno.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <signal.h>
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#include <fcntl.h>
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif


#include "udm_common.h"
#include "udm_utils.h"
#include "udm_db.h"
#include "udm_agent.h"
#include "udm_env.h"
#include "udm_conf.h"
#include "udm_services.h"
#include "udm_sdp.h"
#include "udm_log.h"
#include "udm_xmalloc.h"
#include "udm_doc.h"
#include "udm_result.h"
#include "udm_searchtool.h"
#include "udm_vars.h"
#include "udm_filter.h"

/* This should be last include */
#ifdef DMALLOC
#include "dmalloc.h"
#endif

#ifndef INADDR_NONE
#define INADDR_NONE ((unsigned long)-1)
#endif

/********************* SIG Handlers ****************/
static void sighandler(int sign);
static void init_signals(void){
	/* Set up signals handler*/
	UdmSignal(SIGPIPE, sighandler);
	UdmSignal(SIGHUP, sighandler);
	UdmSignal(SIGINT, sighandler);
	UdmSignal(SIGTERM, sighandler);
	UdmSignal(SIGCHLD, sighandler);
	UdmSignal(SIGALRM, sighandler);
}

static int have_sighup=0;
static int have_sigint=0;
static int have_sigterm=0;
static int have_sigpipe = 0;

static void sighandler(int sign){
#ifdef UNIONWAIT
	union wait status;
#else
	int status;
#endif
	switch(sign){
		case SIGPIPE:
			have_sigpipe = 1;
			break;
		case SIGCHLD:
			while (waitpid(-1, &status, WNOHANG) > 0);
			break;
		case SIGHUP:
			have_sighup=1;
			break;
		case SIGINT:
			have_sigint=1;
			break;
		case SIGTERM:
			have_sigterm=1;
			break;
	        case SIGALRM:
		        _exit(0);
			break;
		default:
			break;
	}
	init_signals();
}

/*************************************************************/

static ssize_t UdmSearchdSendPacket(int fd,const UDM_SEARCHD_PACKET_HEADER *hdr,const void *data){
	ssize_t nsent;
	
	nsent=UdmSend(fd,hdr,sizeof(*hdr),0);
	if(nsent!=sizeof(*hdr))
		return(nsent);
	if(data){
		nsent+=UdmSend(fd,data,hdr->len,0);
	}
	return nsent;
}

static int do_client(UDM_ENV * Conf, int client){
	UDM_SEARCHD_PACKET_HEADER hdr;
	UDM_AGENT   *Agent;
	UDM_RESULT  *Res;
	char buf[1024]="";
	char *words = NULL;
	ssize_t nrecv,nsent;
	int verb=-1;
	int done=0;
	const char *bcharset;
	
	Agent=UdmAgentInit(NULL,Conf,0);
	
	Res=UdmResultInit(NULL);
	
	while(!done){
		size_t dlen=0,ndocs,i;
		int * doc_id=NULL;
		char * dinfo=NULL;
		
		UdmLog(Agent,verb,"Waiting for command header");
		nrecv=UdmRecvall(client,&hdr,sizeof(hdr));
		if(nrecv!=sizeof(hdr)){
			UdmLog(Agent,verb,"Received incomplete header nrecv=%d",(int)nrecv);
			break;
		}else{
			UdmLog(Agent,verb,"Received header cmd=%d len=%d",hdr.cmd,hdr.len);
		}
		switch(hdr.cmd){
			case UDM_SEARCHD_CMD_DOCINFO:
				ndocs=hdr.len/sizeof(*doc_id);
				UdmLog(Agent,verb,"Received DOCINFO command len=%d ndocs=%d",hdr.len,ndocs);
				doc_id=(int*)malloc((hdr.len+1)*sizeof(*doc_id));
				nrecv=UdmRecvall(client,doc_id,hdr.len);
				if(nrecv!=hdr.len){
					UdmLog(Agent,verb,"Received incomplete data nbytes=%d nrecv=%d",hdr.len,(int)nrecv);
					UDM_FREE(doc_id);
					done=1;
					break;
				}
				
				
				Res->num_rows=ndocs;
				Res->Doc=(UDM_DOCUMENT*)malloc(sizeof(UDM_DOCUMENT)*(Res->num_rows));
				
				for(i=0;i<ndocs;i++){
					UdmDocInit(&Res->Doc[i]);
					UdmVarListReplaceInt(&Res->Doc[i].Sections,"ID",doc_id[i]);
				}
				UDM_FREE(doc_id);
				
				if(UDM_OK!=UdmResAction(Agent,Res,UDM_RES_ACTION_DOCINFO,Agent->Conf->db)){
					UdmResultFree(Res);
					snprintf(buf,sizeof(buf)-1,"%s",UdmEnvErrMsg(Agent->Conf));
					UdmLog(Agent,verb,"%s",UdmEnvErrMsg(Agent->Conf));
					hdr.cmd=UDM_SEARCHD_CMD_ERROR;
					hdr.len=strlen(buf);
					nsent=UdmSearchdSendPacket(client,&hdr,buf);
					done=1;
					break;
				}
				
				dlen=0;
				
				for(i=0;i<Res->num_rows;i++){
					size_t		ulen;
					size_t		olen;
					char		textbuf[10*1024];
					size_t		nsec;
					UDM_DOCUMENT	*D=&Res->Doc[i];
					
					for(nsec=0;nsec<D->Sections.nvars;nsec++)
						D->Sections.Var[nsec].section=1;
					
					UdmDocToTextBuf(D,textbuf,sizeof(textbuf)-1);
					textbuf[sizeof(textbuf)-1]='\0';
					
					ulen=strlen(textbuf)+2;
					olen=dlen;
					dlen=dlen+ulen;
					dinfo=(char*)realloc(dinfo,dlen+1);
					dinfo[olen]='\0';
					sprintf(dinfo+olen,"%s\r\n",textbuf);
				}
				
				if(!dinfo)dinfo=strdup("nodocinfo");
				
				hdr.cmd=UDM_SEARCHD_CMD_DOCINFO;
				hdr.len=strlen(dinfo);
				nsent=UdmSearchdSendPacket(client,&hdr,dinfo);
				UdmLog(Agent,verb,"Sent doc_info packet %d bytes",(int)nsent);
				UDM_FREE(dinfo);
				
				break;
				
			case UDM_SEARCHD_CMD_WORDS:
			        words = (char*)realloc(words, hdr.len + 1);
				if (words == NULL) {
					snprintf(buf, sizeof(buf)-1, "Can't alloc memory for query");
					UdmLog(Agent, verb, "Can't alloc memory for query");
					hdr.cmd=UDM_SEARCHD_CMD_ERROR;
					hdr.len=strlen(buf);
					UdmSearchdSendPacket(client,&hdr,buf);
					done=1;
					break;
				}
				nrecv=UdmRecvall(client, words, hdr.len); /* FIXME: check */
				words[nrecv]='\0';
				UdmLog(Agent,verb,"Received words len=%d words='%s'",nrecv,words);
				
				UdmParseQueryString(Agent,&Agent->Conf->Vars,words);
				bcharset = UdmVarListFindStr(&Agent->Conf->Vars,"BrowserCharset","iso-8859-1");
				if (!(Agent->Conf->bcs = UdmGetCharSet(bcharset))) {
					snprintf(buf,sizeof(buf)-1,"Unknown BrowserCharset: %s", bcharset);
					UdmLog(Agent,verb,"Unknown BrowserCharset: %s", bcharset);
					hdr.cmd=UDM_SEARCHD_CMD_ERROR;
					hdr.len=strlen(buf);
					UdmSearchdSendPacket(client,&hdr,buf);
					done=1;
					break;
				}
				
				UdmLog(Agent, verb, "Query: %s [Charset: %s]", 
					UdmVarListFindStr(&Agent->Conf->Vars,"q",""), bcharset);
				
				UdmPrepare(Agent, Res);		/* Prepare search query */
				UdmResAction(Agent,Res,UDM_RES_ACTION_WORDS,Agent->Conf->db);
				
				if(UdmEnvErrCode(Agent->Conf)){
					snprintf(buf,sizeof(buf)-1,"%s",UdmEnvErrMsg(Agent->Conf));
					UdmLog(Agent,verb,"%s",UdmEnvErrMsg(Agent->Conf));
					hdr.cmd=UDM_SEARCHD_CMD_ERROR;
					hdr.len=strlen(buf);
					UdmSearchdSendPacket(client,&hdr,buf);
					done=1;
					break;
				}
				
				snprintf(buf,sizeof(buf)-1,"Total_found=%d",Res->total_found);
				hdr.cmd=UDM_SEARCHD_CMD_MESSAGE;
				hdr.len=strlen(buf);
				nsent=UdmSearchdSendPacket(client,&hdr,buf);
				UdmLog(Agent,verb,"Sent total_found packet %d bytes buf='%s'",(int)nsent,buf);

				if (Res->offset) {
				  hdr.cmd = UDM_SEARCHD_CMD_WITHOFFSET;
				  hdr.len = 0;
				  nsent = UdmSearchdSendPacket(client,&hdr,buf);
				  UdmLog(Agent,verb,"Sent withoffset packet %d bytes",(int)nsent);
				}


				{
				  char *wbuf, *p;
				  hdr.cmd = UDM_SEARCHD_CMD_WWL;
				  hdr.len = sizeof(UDM_WIDEWORDLIST);
				  for (i = 0; i < Res->WWList.nwords; i++) {
				    hdr.len += sizeof(UDM_WIDEWORD) 
				      + (sizeof(int) + sizeof(char)) * (Res->WWList.Word[i].len + 1) 
				      + sizeof(int);
				  }
				  p = wbuf = (char *)UdmXmalloc(hdr.len); /* need UdmXmalloc */
				  if (p != NULL) {
				    memcpy(p, &(Res->WWList), sizeof(UDM_WIDEWORDLIST));
				    p += sizeof(UDM_WIDEWORDLIST);
				  }
				  for (i = 0; i < Res->WWList.nwords; i++) {
				    memcpy(p, &(Res->WWList.Word[i]), sizeof(UDM_WIDEWORD));
				    p += sizeof(UDM_WIDEWORD);
				    memcpy(p, Res->WWList.Word[i].word, Res->WWList.Word[i].len);
				    p += Res->WWList.Word[i].len + 1;
				    p += sizeof(int) - ((int)p % sizeof(int));
				    memcpy(p, Res->WWList.Word[i].uword, sizeof(int)*Res->WWList.Word[i].len);
				    p += sizeof(int) * (Res->WWList.Word[i].len + 1);
				  }
				  nsent = UdmSearchdSendPacket(client, &hdr, wbuf);
				  UdmLog(Agent, verb, "Sent WWL packet %d bytes cmd=%d len=%d nwords=%d", 
					 (int)nsent, hdr.cmd, hdr.len, Res->WWList.nwords);

				  UDM_FREE(wbuf);
				}

				hdr.cmd=UDM_SEARCHD_CMD_WORDS;
				hdr.len=Res->CoordList.ncoords*sizeof(UDM_URL_CRD);
				nsent=UdmSearchdSendPacket(client,&hdr,Res->CoordList.Coords);
				UdmLog(Agent,verb,"Sent words packet %d bytes cmd=%d len=%d nwords=%d",(int)nsent,hdr.cmd,hdr.len,Res->CoordList.Coords);
				/*
				for(i=0;i<Agent->total_found;i++){
					snprintf(buf,sizeof(buf)-1,"u=%08X c=%08X",wrd[i].url_id,wrd[i].coord);
					UdmLog(Agent,verb,"%s",buf);
				}
				*/
				break;
			case UDM_SEARCHD_CMD_GOODBYE:
				UdmLog(Agent,verb,"Received goodbye command",hdr.cmd);
				done=1;
				break;
			default:
				UdmLog(Agent,verb,"Unknown command %d",hdr.cmd);
				done=1;
				break;
		}
	}
	close(client);
	client=0;
	UdmLog(Agent,verb,"Quit");
	UdmAgentFree(Agent);
	UdmResultFree(Res);
	UDM_FREE(words);
	return 0;
}

/*************************************************************/

static void usage(void){

	fprintf(stderr,
"\n\
searchd from %s-%s-%s\n\
http://www.mnogosearch.org (C) 1998-2002, LavTech Corp.\n\
\n\
Usage: searchd [OPTIONS]\n\
\n\
Options are:\n\
  -l		do not log to stderr\n\
  -v n          verbose level, 0-5\n\
  -h,-?         print this help page and exit\n\
\n\
Please post bug reports and suggestions to http://www.mnogosearch.org/bugs/\n",
	PACKAGE,VERSION,UDM_DBTYPE);

	return;
}

#define SEARCHD_EXIT    0
#define SEARCHD_RELOAD  1

static int conf_main(int ctl_sock,UDM_AGENT * Agent){
	fd_set mask;
	int mdone=0;
	int verb=-1;
	int res=SEARCHD_EXIT;
	char reason[1024]="";

	FD_ZERO(&mask);
	FD_SET(ctl_sock,&mask);

	while(!mdone) {
		pid_t fres;
		int sel;
		struct timeval tval;
		fd_set msk;
		
		tval.tv_sec=300;
		tval.tv_usec=0;
		msk=mask;
		sel=select(16,&msk,0,0,&tval);

		if(have_sighup){
			UdmLog(Agent,verb,"SIGHUP arrived");
			have_sighup=0;
			res=SEARCHD_RELOAD;
			mdone=1;
		}
		if(have_sigint){
			UdmLog(Agent,verb,"SIGINT arrived");
			have_sigint=0;
			mdone=1;
		}
		if(have_sigterm){
			UdmLog(Agent,verb,"SIGTERM arrived");
			have_sigterm=0;
			mdone=1;
		}
		if(have_sigpipe){
			UdmLog(Agent, verb, "SIGPIPE arrived. Broken pipe!");
			have_sigpipe = 0;
			mdone = 1;
		}

		if(mdone)break;
		
		if(sel==0)continue;
		if(sel==-1){
			switch(errno){
				case EINTR:	/* Child */
					break;
				default:
					UdmLog(Agent,verb,"FIXME select error %d %s",errno,strerror(errno));
			}
			continue;
		}
		
		if(FD_ISSET(ctl_sock,&msk)){
			int ns;
			struct sockaddr_in client_addr;
			socklen_t addrlen=sizeof(client_addr);
			char addr[128]="";
			int method;
			
			if ((ns = accept(ctl_sock, (struct sockaddr *) &client_addr, &addrlen)) == -1) {
				UdmLog(Agent,verb,"accept() error %d %s",errno,strerror(errno));
				UdmEnvFree(Agent->Conf);
				UdmAgentFree(Agent);
				exit(1);
			}
			UdmLog(Agent,verb,"Connect %s", inet_ntoa(client_addr.sin_addr));
			
			snprintf(addr,sizeof(addr)-1,inet_ntoa(client_addr.sin_addr));
			method=UdmFilterFind(&Agent->Conf->Filters,addr,reason);
			
			UdmLog(Agent,verb,"%s",reason);
			
			if(method==UDM_METHOD_DISALLOW){
				UdmLog(Agent,verb,"Reject client");
				close(ns);
				continue;
			}
			
			if((fres=fork())==0){
				alarm(300); /* 5 min. - maximum time of child execution */
				init_signals();
				
				closesocket(ctl_sock);
				do_client(Agent->Conf,ns);
				UdmEnvFree(Agent->Conf);
				UdmAgentFree(Agent);
				exit(0);
			}else{
				close(ns);
				if(fres>0){
					
				}else{
					UdmLog(Agent,verb,"fork error %d",fres);
				}
			}
			
		}
	}
	return res;
}

int main(int argc, char **argv) {
	int done=0;
	int ch=0;
	int nport=UDM_SEARCHD_PORT;
	int log2stderr=1;
	const char * config_name = UDM_CONF_DIR "/searchd.conf";
	int debug_level = UDM_LOG_INFO;

	UdmInit(); /* Initialize library */

	while ((ch = getopt(argc, argv, "l?v:")) != -1){
		switch (ch) {
			case 'l':
				log2stderr=0;
				break;
		        case 'v': debug_level = atoi(optarg); break;
			case 'h':
			case '?':
			default:
				usage();
				return 1;
				break;
		}
	}
	argc -= optind;argv += optind;
	if(argc==1)config_name=argv[0];


	while(!done){
		struct sockaddr_in server_addr;
		int ctl_sock, on=1;
		UDM_AGENT * Agent;
		UDM_ENV * Conf=NULL;
		int verb=-1;
		int flags=UDM_FLAG_SPELL;
		int res=0;
		const char *lstn;

		Conf = UdmEnvInit(NULL);
		Agent=UdmAgentInit(NULL,Conf,0);
		
		res=UdmLoadConfig(Conf,config_name,0,flags);
		
		UdmOpenLog("searchd",Conf,log2stderr);
		
		UdmSetLogLevel(Conf, debug_level);

		if(res!=UDM_OK){
			UdmLog(Agent,verb,"%s",UdmEnvErrMsg(Conf));
			UdmAgentFree(Agent);
			UdmEnvFree(Conf);
			exit(1);
		}

		UdmLog(Agent,verb,"searchd started with '%s'",config_name);
		UdmLog(Agent,verb,"VarDir: '%s'",Agent->Conf->vardir);
		UdmLog(Agent,verb,"Affixes: %d, Spells: %d, Synonyms: %d",Conf->Affixes.naffixes,Conf->Spells.nspell,Conf->Synonyms.nsynonyms);
		
		if ((ctl_sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
			UdmLog(Agent,verb,"socket() error %d",errno);
			UdmAgentFree(Agent);
			UdmEnvFree(Conf);
			exit(1);
		}

		if (setsockopt(ctl_sock, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on)) != 0){
			UdmLog(Agent,verb,"setsockopt() error %d",errno);
			UdmAgentFree(Agent);
			UdmEnvFree(Conf);
			exit(1);
		}

		bzero(&server_addr, sizeof(server_addr));
		server_addr.sin_family = AF_INET;
		if((lstn=UdmVarListFindStr(&Agent->Conf->Vars,"Listen",NULL))){
			char * cport;
			
			if((cport=strchr(lstn,':'))){
				UdmLog(Agent,verb,"Listening '%s'",lstn);
				*cport='\0';
				server_addr.sin_addr.s_addr = inet_addr(lstn);
				nport=atoi(cport+1);
			}else{
				nport=atoi(lstn);
				server_addr.sin_addr.s_addr = htonl(INADDR_ANY);
				UdmLog(Agent,verb,"Listening port %d",nport);
			}
		}else{
			UdmLog(Agent,verb,"Listening port %d",nport);
			server_addr.sin_addr.s_addr = htonl(INADDR_ANY);
		}
		server_addr.sin_port = htons((u_short)nport);

		if (bind(ctl_sock, (struct sockaddr *)&server_addr, sizeof(server_addr)) == -1) {
			UdmLog(Agent,verb,"Can't bind: error %d %s",errno,strerror(errno));
			UdmAgentFree(Agent);
			UdmEnvFree(Conf);
			exit(1);
		}

		/* Backlog 32 is enough */
		if (listen(ctl_sock, 32) == -1) {
			UdmLog(Agent,verb,"listen() error %d %s",errno,strerror(errno));
			UdmAgentFree(Agent);
			UdmEnvFree(Conf);
			exit(1);
		}
		
		UdmLog(Agent,verb,"Ready");
		
		init_signals();
		res=conf_main(ctl_sock,Agent);
		closesocket(ctl_sock);

		if(res!=SEARCHD_RELOAD){
			done=1;
			UdmLog(Agent,verb,"Shutdown");
		}else{
			UdmLog(Agent,verb,"Reloading conf");
		}
		UdmAgentFree(Agent);
		UdmEnvFree(Conf);
	}
		
	return 0;
}
